#include "TalonFiveGameInputMapping.h"
#include "TalonFiveModule.h"
#include "Resolver.h"
#include "TalonFiveConstants.h"
#include "Settings.h"
#include "SettingsGetValueT.h"
#include "Utility.h"
#define GLFW_DLL
#include <GL/glfw.h>
#include "GlfwConstants.h"
#include "TalonFiveGameInputMapping.h"
#include "Game/InputObject.h"

#include <math.h>
#include <string>
#include <fstream>


namespace TalonFive {

	void Module::Init()
	{
		//[12:51] <@deleter> if you create any sort of player or enemy objects, make sure they inherit from IGameObject and ISyncObject

		glGenVertexArrays(1, &vertexArrayID);
		glBindVertexArray(vertexArrayID);

		bgShaderID = LoadShaders("bg.vs", "bg.fs");
		fieldShaderID = LoadShaders("field.vs", "field.fs");
		starShaderID = LoadShaders("star.vs", "star.fs");

		matrixID = glGetUniformLocation(fieldShaderID, "MVP");
		bgTextureID = glGetUniformLocation(bgShaderID, "bgTex");
		//timeID = glGetUniformLocation(starShaderID, "time");

		int xres = GET_SETTING(basecode::Game::Glfw::resx, int);
		int yres = GET_SETTING(basecode::Game::Glfw::resy, int);

		float fovy = 45;
		float aspect = xres / (float)yres;
		projectionMatrix = glm::perspective(fovy, aspect, .1f, 100.0f);

		viewMatrix = glm::lookAt(glm::vec3(0, 0, 80), glm::vec3(0, 0, -1), glm::vec3(0, 1, 0));
		
		glViewport(0, 0, xres, yres);

		bgTex = loadDDS("data/stars.DDS");

		static const GLfloat bgQuad[] = {
                -1.0f, -1.0f, 0.0f,
                 1.0f, -1.0f, 0.0f,
                -1.0f,  1.0f, 0.0f,
                -1.0f,  1.0f, 0.0f,
                 1.0f, -1.0f, 0.0f,
                 1.0f,  1.0f, 0.0f,
        };

        glGenBuffers(1, &bgVertexBuffer);
        glBindBuffer(GL_ARRAY_BUFFER, bgVertexBuffer);
        glBufferData(GL_ARRAY_BUFFER, sizeof(bgQuad), bgQuad, GL_STATIC_DRAW);

		float uvw = (1 - yres / (float)xres) / 2.0f;
		
		static const GLfloat bgUVs[] = {
                 0.0f,  uvw,
                 1.0f,  uvw,
                 0.0f,  1.0f - uvw,
                 0.0f,  1.0f - uvw,
                 1.0f,  uvw,
                 1.0f,  1.0f - uvw,
        };

        glGenBuffers(1, &bgUVBuffer);
        glBindBuffer(GL_ARRAY_BUFFER, bgUVBuffer);
        glBufferData(GL_ARRAY_BUFFER, sizeof(bgUVs), bgUVs, GL_STATIC_DRAW);
		
		
		timer.reset();
		fps.onSeconds((float).2);
		//font.load("data/pf_ronda_seven_bold.fnt", 10);
		
		ad_x = basecode::ActiveDebug('F', 'S');
		ad_y = basecode::ActiveDebug('E', 'D');
		ad_z = basecode::ActiveDebug('R', 'W');

		srand((unsigned int)time(NULL));

		basecode::Physics::PhysicsObject* po;
		basecode::Model pmodel;

		// create the player
		PlayerObject* player = new PlayerObject();
		po = player->physics();
		po->position.set(30, 0, 0);

		pmodel.load("data/player_hull.obj");
		pmodel.center();
		pmodel.normalize();
		pmodel.scale(2);

		po->hull.importModel(pmodel);
		po->orientation.loadIdentity();
		po->orientation.rotate(90, 0, 0);
		po->color.set(1);
		po->setMass(50);

		objList.push_back(player);
		//////

		Gamepad gpad;
		for (int i = 0; i < 16; i++) gamepad.push_back(gpad);
		shipAngle = 0;
		
		ptimer.init(&timer);
		
		massDensity = 1;

		
		glLineWidth(2);

		glGenBuffers(1, &starVertexBuffer);
		glGenBuffers(1, &starColorBuffer);

		std::vector<float> strVtx;
		std::vector<float> strClr;

		float r = .0005f;

		Star star;
		for (unsigned int i = 0; i < 1000; i++)
		{
			star.position.set(basecode::rnd() * 2 - 1, basecode::rnd() * 2 - 1);
			star.c.set(1 - basecode::rnd() * .25f, 1 - basecode::rnd() * .25f, 1, basecode::rnd());
			starList.push_back(star);

			//star.c.scale(.75f);

			for (int j = 0; j < 24; j++)
			{
				strVtx.push_back(star.position.x);
				strVtx.push_back(star.position.y);
				strVtx.push_back(0);

				strVtx.push_back(star.position.x + (1 + i % 5) * r * cos(3.14159f * 2 * j / 12.0f));
				strVtx.push_back(star.position.y + (1 + i % 5) * r * aspect * sin(3.14159f * 2 * j / 12.0f));
				strVtx.push_back(0);

				strVtx.push_back(star.position.x + (1 + i % 5) * r * cos(3.14159f * 2 * (j + 1) / 12.0f));
				strVtx.push_back(star.position.y + (1 + i % 5) * r * aspect * sin(3.14159f * 2 * (j + 1) / 12.0f));
				strVtx.push_back(0);

				strClr.push_back(star.c.r);
				strClr.push_back(star.c.g);
				strClr.push_back(star.c.b);
				strClr.push_back(1);

				strClr.push_back(star.c.r);
				strClr.push_back(star.c.g);
				strClr.push_back(star.c.b);
				strClr.push_back(0);

				strClr.push_back(star.c.r);
				strClr.push_back(star.c.g);
				strClr.push_back(star.c.b);
				strClr.push_back(0);

			}
		}

		glBindBuffer(GL_ARRAY_BUFFER, starVertexBuffer);
		glBufferData(GL_ARRAY_BUFFER, strVtx.size() * sizeof(float), &strVtx[0], GL_STATIC_DRAW);

		glBindBuffer(GL_ARRAY_BUFFER, starColorBuffer);
		glBufferData(GL_ARRAY_BUFFER, strClr.size() * sizeof(float), &strClr[0], GL_STATIC_DRAW);

		/*

		for (unsigned int i = 0; i < starList.size(); i++)
		{
			c = starList[i].c;
			c.scale(.7f + .3f * basecode::rnd());

			for (int j = 0; j < 24; j++)
			{
				c.glColor4f();
				glVertex2fv(&starList[i].position.x);
				glColor4f(c.r, c.g, c.b, 0);
				glVertex2f(starList[i].position.x + (1 + i % 5) * r * cos(3.14159f * 2 * j / 12.0f), starList[i].position.y + (1 + i % 5) * r * sin(3.14159f * 2 * j / 12.0f));
				glVertex2f(starList[i].position.x + (1 + i % 5) * r * cos(3.14159f * 2 * (j + 1) / 12.0f), starList[i].position.y + (1 + i % 5) * r * sin(3.14159f * 2 * (j + 1) / 12.0f));*/
		
		


		float xr, yr, zr;
		
		basecode::Model model;
		basecode::Math::vec3f vx, vy, vz;
		basecode::vec3f pos;

		// create station center
		printf("Generating Station\n");

		StationObject* sobj;
		basecode::Model centerModel;
		centerModel.load("data/center_hull.obj");
		centerModel.center();
		//centerModel.normalize();
		centerModel.scale(2);

		model.load("data/center.obj");
		model.center(); 
		//model.normalize();
		model.scale(2);
		model.calc_normals();
		model.genBuffers();
		model.buildBuffers();
		model.color(1, 1, 1);
		modelList.push_back(model);
		model.clear();

		sobj = new StationObject();
		sobj->modelListPtr = &modelList;
		sobj->set_modelid(0);

		stationObj = sobj;

		po = sobj->physics();

		po->hull.importModel(centerModel);
		po->orientation.loadIdentity();
		po->orientation.rotate(90, 0, 0);
		po->color.set(1);
		po->setMass(5000000);
					
		po->position.set(0, 0, 0);

		objList.push_back(sobj);
		
		// create random truss pieces
		printf("Generating Trusses\n");

		basecode::Model trussModel;
		trussModel.load("data/truss_hull.obj");
		trussModel.center();
		//trussModel.normalize();
		trussModel.scale(2);

		model.load("data/truss.obj");
		model.center();
		//model.normalize();
		model.scale(2);
		model.calc_normals();
		model.genBuffers();
		model.buildBuffers();
		model.color(1, 1, 1);
		modelList.push_back(model);
		model.clear();

		for (unsigned int a = 0; a < 1; a++)
		{
			float mindist2;
			float dist2;
			float radius;
			bool pass;
			unsigned int cnt = 0;
			
			do
			{
				cnt++;
				radius = 5;
				
				pos.set(basecode::rnd() * 200 - 100, basecode::rnd() * 150 - 75, 0);	
				pass = true;
				for (unsigned int b = 0; b < objList.size(); b++)
				{
					dist2 = (pos - objList[b]->physics()->position).magnitude2();
					mindist2 = pow(radius + objList[b]->physics()->hull.radius, 2.0f);
					if (dist2 < mindist2) pass = false;
				}
			} while(!pass && cnt < 1000);
			if (cnt == 1000) break;

			sobj = new StationObject();
			sobj->modelListPtr = &modelList;
			sobj->set_modelid(1);

			po = sobj->physics();
			
			po->position = pos;

			po->hull.importModel(trussModel);
			po->hull.radius = trussModel.radius();
			po->hull.surfaceArea = trussModel.surface_area();

			po->setMass(po->hull.surfaceArea * massDensity);
			po->linearMomentum = basecode::Math::vec3f(basecode::rnd() - .5f, basecode::rnd() - .5f, 0);
			po->linearMomentum.normalize();
			po->linearMomentum.scale(basecode::rnd() * 100);
			
			po->orientation.loadIdentity();
			po->orientation.rotate(90, 0, 0);
			//po.orientation.orient(vx, vy, vz);

			objList.push_back(sobj);
		}



		// create random truss pieces
		printf("Generating Enemies\n");

		EnemyObject* eobj;
		basecode::Model enemyModel;
		enemyModel.load("data/enemy_hull.obj");
		enemyModel.center();
		enemyModel.normalize();
		enemyModel.scale(2);

		for (unsigned int a = 0; a < 1; a++)
		{
			float mindist2;
			float dist2;
			float radius;
			bool pass;
			unsigned int cnt = 0;
			
			do
			{
				cnt++;
				radius = 5;
				
				pos.set(basecode::rnd() * 200 - 100, basecode::rnd() * 150 - 75, 0);	
				pass = true;
				for (unsigned int b = 0; b < objList.size(); b++)
				{
					dist2 = (pos - objList[b]->physics()->position).magnitude2();
					mindist2 = pow(radius + objList[b]->physics()->hull.radius, 2.0f);
					if (dist2 < mindist2) pass = false;
				}
			} while(!pass && cnt < 1000);
			if (cnt == 1000) break;

			eobj = new EnemyObject();
			
			po = eobj->physics();
			
			po->position = pos;

			po->hull.importModel(enemyModel);
			po->hull.radius = enemyModel.radius();
			po->hull.surfaceArea = enemyModel.surface_area();

			po->setMass(po->hull.surfaceArea * massDensity);
			po->linearMomentum = basecode::Math::vec3f(basecode::rnd() - .5f, basecode::rnd() - .5f, 0);
			po->linearMomentum.normalize();
			po->linearMomentum.scale(basecode::rnd() * 100);
			
			po->orientation.loadIdentity();
			vx.set(rnd() * 2 - 1, rnd() * 2 - 1, rnd() * 2 - 1);
			vy.set(rnd() * 2 - 1, rnd() * 2 - 1, rnd() * 2 - 1);
			vz = vx.cross(vy);
			vy = vz.cross(vx);
			vx.normalize();
			vy.normalize();
			vz.normalize();

			//po.orientation.orient(vx, vy, vz);

			objList.push_back(eobj);
		}



		// create asteroid
		printf("Generating Asteroids\n");

		AsteroidObject* asteroid;
		
		for (unsigned int a = 0; a < 80; a++)
		{
			float mindist2;
			float dist2;
			float radius;
			bool pass;
			unsigned int cnt = 0;
			do
			{
				cnt++;
				xr = 1 + basecode::rnd() * 4;
				yr = 1 + basecode::rnd() * 4;
				zr = 1 + basecode::rnd() * 4;
				radius = sqrt(xr*xr + yr*yr + zr*zr);
				
				pos.set(basecode::rnd() * 200 - 100, basecode::rnd() * 200 - 100, 0);	
				pass = true;
				for (unsigned int b = 0; b < objList.size(); b++)
				{
					dist2 = (pos - objList[b]->physics()->position).magnitude2();
					mindist2 = pow(radius + objList[b]->physics()->hull.radius, 2.0f);
					if (dist2 < mindist2) pass = false;
				}
			} while(!pass && cnt < 1000);
			if (cnt == 1000) break;

			float x, y, z, phi, theta;
			model.clear();
			for (unsigned int i = 0; i < 90; i++)
			{
				phi = 3.14159f * basecode::rnd();
				theta = 3.14159f * 2 * basecode::rnd();

				x = (basecode::rnd() * xr * 2 - xr) * sin(phi) * cos(theta);
				y = (basecode::rnd() * yr * 2 - yr) * sin(phi) * sin(theta);
				z = (basecode::rnd() * zr * 2 - zr) * cos(phi);

				model.newVertex(x, y, z);
			}

			model.center();
			do
			{
				model.convex_hull(16);
			} while (model.faces.size() < 10);

			model.prune_vertices();
			model.center();
			
			asteroid = new AsteroidObject();
			asteroid->setStation(stationObj);

			po = asteroid->physics();
			
			po->position = pos;

			po->hull.importModel(model);
			po->hull.radius = model.radius();
			po->hull.surfaceArea = model.surface_area();

			po->setMass(po->hull.surfaceArea * massDensity);
			po->linearMomentum = basecode::Math::vec3f(basecode::rnd() - .5f, basecode::rnd() - .5f, 0);
			po->linearMomentum.normalize();
			po->linearMomentum.scale(basecode::rnd() * 100);
			
			po->orientation.loadIdentity();
			vx.set(rnd() * 2 - 1, rnd() * 2 - 1, rnd() * 2 - 1);
			vy.set(rnd() * 2 - 1, rnd() * 2 - 1, rnd() * 2 - 1);
			vz = vx.cross(vy);
			vy = vz.cross(vx);
			vx.normalize();
			vy.normalize();
			vz.normalize();

			//po.orientation.orient(vx, vy, vz);

			objList.push_back(asteroid);
		}

		objList[0]->physics()->hull.color(1, 1, 1);
		for (unsigned int i = 1; i < objList.size(); i++) objList[i]->physics()->hull.color(.21f, .21f, .21f);


		float bscale = 0;
		for (unsigned int a = 1; a < objList.size(); a++)
		{
			if (objList[a]->physics()->hull.radius > bscale) bscale = objList[a]->physics()->hull.radius;
		}
		bscale *= 4;
		
		// build physics volumes - dont bother with top and bottom faces
		printf("Building p.Volumes\n");
		float xx, yy, w = bscale;

		std::vector<vec3f> vp;
		basecode::Physics::Volume vol;
		
		for (int x = -6; x < 6; x++)
		{
			for (int y = -6; y < 6; y++)
			{
				xx = x * w;
				yy = y * w;

				vol.clear();
				
				// left face
				vp.clear();
				vp.push_back(basecode::Math::vec3f(xx, yy, -w));
				vp.push_back(basecode::Math::vec3f(xx, yy, w));
				vp.push_back(basecode::Math::vec3f(xx, yy - w, w));
				vp.push_back(basecode::Math::vec3f(xx, yy - w, -w));
				vol.addPlane(vp);

				// top y face
				vp.clear();
				vp.push_back(basecode::Math::vec3f(xx + w, yy, -w));
				vp.push_back(basecode::Math::vec3f(xx + w, yy, w));
				vp.push_back(basecode::Math::vec3f(xx, yy, w));
				vp.push_back(basecode::Math::vec3f(xx, yy, -w));
				vol.addPlane(vp);

				// right face
				vp.clear();
				vp.push_back(basecode::Math::vec3f(xx + w, yy, w));
				vp.push_back(basecode::Math::vec3f(xx + w, yy, -w));
				vp.push_back(basecode::Math::vec3f(xx + w, yy - w, -w));
				vp.push_back(basecode::Math::vec3f(xx + w, yy - w, w));
				vol.addPlane(vp);

				// bottom y face
				vp.clear();
				vp.push_back(basecode::Math::vec3f(xx + w, yy - w, w));
				vp.push_back(basecode::Math::vec3f(xx + w, yy - w, -w));
				vp.push_back(basecode::Math::vec3f(xx, yy - w, -w));
				vp.push_back(basecode::Math::vec3f(xx, yy - w, w));
				vol.addPlane(vp);

				volumeList.push_back(vol);
			}
		}

		asel = 0;
		

		printf("Initialization Complete\n");
	}

	// move this somewhere else
	Gamepad::Gamepad()
	{
		connected = false;
		for (int d = 0; d < 16; d++)
		{
			minValue[d] = 0;
			maxValue[d] = 0;
		}
	}

	// move this somewhere else
	void Module::SplitModel(basecode::Model* mSrc, basecode::Model* mDstA, basecode::Model* mDstB, basecode::Math::vec3f split_n, basecode::Math::vec3f split_p)
	{
	
		mDstA->clear();
		mDstB->clear();

		std::vector<basecode::Math::vec2i> edge_split;
		std::vector<basecode::Math::vec2i> edge_list;

		basecode::Math::vec2i edge;
		bool pass;

		for (unsigned int ff = 0; ff < mSrc->faces.size(); ff++)
		{
			for (unsigned int p = 0; p < 3; p++)
			{
				edge.set(mSrc->faces[ff].point_id[p], mSrc->faces[ff].point_id[(p + 1) % 3]);
				pass = true;
				for (unsigned int e = 0; e < edge_list.size(); e++)
				{
					if ((edge_list[e].x == edge.x && edge_list[e].y == edge.y) || (edge_list[e].y == edge.x && edge_list[e].x == edge.y))
					{
						pass = false;
						break;
					}
				}
				if (pass) edge_list.push_back(edge);
			}
		}

		//for (unsigned int e = 0; e < edge_list.size(); e++)

		float dot0, dot1;


		edge_split.clear();
		for (unsigned int e = 0; e < edge_list.size(); e++)
		{
			pass = false;

			dot0 = split_n.dot(mSrc->vertices[edge_list[e].x] - split_p);
			dot1 = split_n.dot(mSrc->vertices[edge_list[e].y] - split_p);
			if (dot0 < 0 && dot1 < 0) continue;
			if (dot0 > 0 && dot1 > 0) continue;
			if (fabs(dot0) < .001f) continue;			
			if (fabs(dot1) < .001f) continue;

			edge_split.push_back(edge_list[e]);
		}


		basecode::vec3f v, p0, n;
		float t;
		unsigned int tpnt;

		n = split_n;
		for (unsigned int e = 0; e < edge_split.size(); e++)
		{
			// find point at line-plane intersection
			p0 = mSrc->vertices[edge_split[e].x];
			v = mSrc->vertices[edge_split[e].y] - p0;
			t = (split_p - p0).dot(n) / v.dot(n);
			p0 = p0 + v * t;
			tpnt = mSrc->newVertex(p0);

			// locate faces that contain this edge
			for (unsigned int ff = 0; ff < mSrc->faces.size(); ff++)
			{
				for (unsigned int p = 0; p < 3; p++)
				{
					edge.set(mSrc->faces[ff].point_id[p], mSrc->faces[ff].point_id[(p + 1) % 3]);
					if ((edge_split[e].x == edge.x && edge_split[e].y == edge.y) || (edge_split[e].y == edge.x && edge_split[e].x == edge.y))
					{
						// found face
						// update one face point to split location
						// add a new face to other half of split
						mSrc->newFace(tpnt, mSrc->faces[ff].point_id[(p + 1) % 3], mSrc->faces[ff].point_id[(p + 2) % 3]);
						mSrc->faces[ff].point_id[(p + 1) % 3] = tpnt;
					}
				}
			}
		}

		
		basecode::Math::vec3f pcntr;
		float pcntr_c = 0;

		// redetermine edges using new cuts
		edge_split.clear();
		edge_list.clear();
		for (unsigned int ff = 0; ff < mSrc->faces.size(); ff++)
		{
			for (unsigned int p = 0; p < 3; p++)
			{
				edge.set(mSrc->faces[ff].point_id[p], mSrc->faces[ff].point_id[(p + 1) % 3]);
				pass = true;
				for (unsigned int e = 0; e < edge_list.size(); e++)
				{
					if ((edge_list[e].x == edge.x && edge_list[e].y == edge.y) || (edge_list[e].y == edge.x && edge_list[e].x == edge.y))
					{
						pass = false;
						break;
					}
				}
				if (pass) edge_list.push_back(edge);
			}
		}

		for (unsigned int e = 0; e < edge_list.size(); e++)
		{
			dot0 = split_n.dot(mSrc->vertices[edge_list[e].x] - split_p);
			dot1 = split_n.dot(mSrc->vertices[edge_list[e].y] - split_p);
			if (fabs(dot0) < .001f && fabs(dot1) < .001f)
			{
				edge_split.push_back(edge_list[e]);
				pcntr += mSrc->vertices[edge_list[e].x];
				pcntr += mSrc->vertices[edge_list[e].y];
				pcntr_c += 2;
			}
		}

		if (pcntr_c > 0)
		{
			// create 2 bodies
			float dot;
			for (unsigned int f = 0; f < mSrc->faces.size(); f++)
			{
				pass = false;
				for (unsigned int p = 0; p < 3; p++)
				{
					dot = split_n.dot(mSrc->vertices[mSrc->faces[f].point_id[p]] - split_p);
					if (dot < -.001f) pass = true;
				}
				if (pass)
				{
					mDstA->newFace(mSrc->vertices[mSrc->faces[f].point_id[0]], mSrc->vertices[mSrc->faces[f].point_id[1]], mSrc->vertices[mSrc->faces[f].point_id[2]]);
				}

				pass = false;
				for (unsigned int p = 0; p < 3; p++)
				{
					dot = split_n.dot(mSrc->vertices[mSrc->faces[f].point_id[p]] - split_p);
					if (dot > .001f) pass = true;
				}
				if (pass)
				{
					mDstB->newFace(mSrc->vertices[mSrc->faces[f].point_id[0]], mSrc->vertices[mSrc->faces[f].point_id[1]], mSrc->vertices[mSrc->faces[f].point_id[2]]);
				}
			}

			pcntr /= pcntr_c;
			
			for (unsigned int e = 0; e < edge_split.size(); e++)
			{
				mDstA->newFace(mSrc->vertices[edge_split[e].x], mSrc->vertices[edge_split[e].y], pcntr);
				mDstB->newFace(mSrc->vertices[edge_split[e].x], mSrc->vertices[edge_split[e].y], pcntr);
			}
			
			mDstA->removeCreases();
			mDstB->removeCreases();

			mDstA->orient_normals(1);
			mDstB->orient_normals(1);

		}
		
	}

	Mouse::Mouse()
	{
		hold = false;
		middle_hold = false;
	}


	TetherBeam::TetherBeam()
	{
		init();
	}

	void TetherBeam::init()
	{
		tetherA = 0;
		tetherB = 0;
		expireTime = 0;
	}

	void TetherBeam::create(basecode::Game::IGameObject* to)
	{
		tetherA = to;
	}

	void TetherBeam::setExpiration(float time)
	{
		expireTime = time;
	}

	float TetherBeam::expiration()
	{
		return expireTime;
	}
	

	void TetherBeam::pair(basecode::Game::IGameObject* to)
	{
		tetherB = to;
	}

	void TetherBeam::disable()
	{
		init();
	}

	void TetherBeam::update(float interval)
	{
		if (tetherA) tetherA->update(interval);
		if (tetherB) tetherB->update(interval);
	}

	GLuint Module::LoadShaders(const char * vertex_file_path,const char * fragment_file_path){
 
    // Create the shaders
    GLuint VertexShaderID = glCreateShader(GL_VERTEX_SHADER);
    GLuint FragmentShaderID = glCreateShader(GL_FRAGMENT_SHADER);
 
    // Read the Vertex Shader code from the file
    std::string VertexShaderCode;
    std::ifstream VertexShaderStream(vertex_file_path, std::ios::in);
    if(VertexShaderStream.is_open())
    {
        std::string Line = "";
        while(getline(VertexShaderStream, Line))
            VertexShaderCode += "\n" + Line;
        VertexShaderStream.close();
    }
 
    // Read the Fragment Shader code from the file
    std::string FragmentShaderCode;
    std::ifstream FragmentShaderStream(fragment_file_path, std::ios::in);
    if(FragmentShaderStream.is_open()){
        std::string Line = "";
        while(getline(FragmentShaderStream, Line))
            FragmentShaderCode += "\n" + Line;
        FragmentShaderStream.close();
    }
 
    GLint Result = GL_FALSE;
    int InfoLogLength;
 
    // Compile Vertex Shader
    printf("Compiling shader : %s\n", vertex_file_path);
    char const * VertexSourcePointer = VertexShaderCode.c_str();
    glShaderSource(VertexShaderID, 1, &VertexSourcePointer , NULL);
    glCompileShader(VertexShaderID);
 
    // Check Vertex Shader
    glGetShaderiv(VertexShaderID, GL_COMPILE_STATUS, &Result);
    glGetShaderiv(VertexShaderID, GL_INFO_LOG_LENGTH, &InfoLogLength);
    std::vector<char> VertexShaderErrorMessage(InfoLogLength);
    glGetShaderInfoLog(VertexShaderID, InfoLogLength, NULL, &VertexShaderErrorMessage[0]);
	if (strlen(&VertexShaderErrorMessage[0]) > 0) fprintf(stdout, "%s\n", &VertexShaderErrorMessage[0]);
 
    // Compile Fragment Shader
    printf("Compiling shader : %s\n", fragment_file_path);
    char const * FragmentSourcePointer = FragmentShaderCode.c_str();
    glShaderSource(FragmentShaderID, 1, &FragmentSourcePointer , NULL);
    glCompileShader(FragmentShaderID);
 
    // Check Fragment Shader
    glGetShaderiv(FragmentShaderID, GL_COMPILE_STATUS, &Result);
    glGetShaderiv(FragmentShaderID, GL_INFO_LOG_LENGTH, &InfoLogLength);
    std::vector<char> FragmentShaderErrorMessage(InfoLogLength);
    glGetShaderInfoLog(FragmentShaderID, InfoLogLength, NULL, &FragmentShaderErrorMessage[0]);
    if (strlen(&FragmentShaderErrorMessage[0]) > 0) fprintf(stdout, "%s\n", &FragmentShaderErrorMessage[0]);
 
    // Link the program
    fprintf(stdout, "Linking program\n");
    GLuint ProgramID = glCreateProgram();
    glAttachShader(ProgramID, VertexShaderID);
    glAttachShader(ProgramID, FragmentShaderID);
    glLinkProgram(ProgramID);
 
    // Check the program
    glGetProgramiv(ProgramID, GL_LINK_STATUS, &Result);
    glGetProgramiv(ProgramID, GL_INFO_LOG_LENGTH, &InfoLogLength);
    if ( InfoLogLength > 0 ){
            std::vector<char> ProgramErrorMessage(InfoLogLength+1);
            glGetProgramInfoLog(ProgramID, InfoLogLength, NULL, &ProgramErrorMessage[0]);
            if (strlen(&ProgramErrorMessage[0]) > 0) printf("%s\n", &ProgramErrorMessage[0]);
    }
 
    glDeleteShader(VertexShaderID);
    glDeleteShader(FragmentShaderID);
 
    return ProgramID;
}

#define FOURCC_DXT1 0x31545844 // Equivalent to "DXT1" in ASCII
#define FOURCC_DXT3 0x33545844 // Equivalent to "DXT3" in ASCII
#define FOURCC_DXT5 0x35545844 // Equivalent to "DXT5" in ASCII

	GLuint Module::loadDDS(const char * imagepath){

		printf("Loading DDS %s\n", imagepath);
 
    unsigned char header[124];
 
    FILE *fp;
 
    /* try to open the file */
    fp = fopen(imagepath, "rb");
    if (fp == NULL)
        return 0;
 
    /* verify the type of file */
    char filecode[4];
    fread(filecode, 1, 4, fp);
    if (strncmp(filecode, "DDS ", 4) != 0) {
        fclose(fp);
        return 0;
    }
 
    /* get the surface desc */
    fread(&header, 124, 1, fp);
 
    unsigned int height      = *(unsigned int*)&(header[8 ]);
    unsigned int width         = *(unsigned int*)&(header[12]);
    unsigned int linearSize     = *(unsigned int*)&(header[16]);
    unsigned int mipMapCount = *(unsigned int*)&(header[24]);
    unsigned int fourCC      = *(unsigned int*)&(header[80]);

	unsigned char * buffer;
    unsigned int bufsize;
    /* how big is it going to be including all mipmaps? */
    bufsize = mipMapCount > 1 ? linearSize * 2 : linearSize;
    buffer = (unsigned char*)malloc(bufsize * sizeof(unsigned char));
    fread(buffer, 1, bufsize, fp);
    /* close the file pointer */
    fclose(fp);

	printf("Read %d bytes\n", bufsize);

	unsigned int components  = (fourCC == FOURCC_DXT1) ? 3 : 4;
    unsigned int format;
    switch(fourCC)
    {
    case FOURCC_DXT1:
        format = GL_COMPRESSED_RGBA_S3TC_DXT1_EXT;
        break;
    case FOURCC_DXT3:
        format = GL_COMPRESSED_RGBA_S3TC_DXT3_EXT;
        break;
    case FOURCC_DXT5:
        format = GL_COMPRESSED_RGBA_S3TC_DXT5_EXT;
        break;
    default:
        free(buffer);
        return 0;
    }

	// Create one OpenGL texture
		GLuint textureID;
		glGenTextures(1, &textureID);

		// "Bind" the newly created texture : all future texture functions will modify this texture
		glBindTexture(GL_TEXTURE_2D, textureID);

		unsigned int blockSize = (format == GL_COMPRESSED_RGBA_S3TC_DXT1_EXT) ? 8 : 16;
		unsigned int offset = 0;
 
		/* load the mipmaps */
		for (unsigned int level = 0; level < mipMapCount && (width || height); ++level)
		{
			unsigned int size = ((width+3)/4)*((height+3)/4)*blockSize;
			glCompressedTexImage2D(GL_TEXTURE_2D, level, format, width, height, 
				0, size, buffer + offset);
 
			offset += size;
			width  /= 2;
			height /= 2;
		}
		free(buffer);
 
		return textureID;
	}

	
}
