#ifndef __ITEM_H__
#define __ITEM_H__

#include <boost/unordered_map.hpp>
#include <string>
#include <sstream>

#define string_t std::string

template <typename T>
string_t NumberToString ( T Number )
{
   std::stringstream ss;
   ss << Number;
   return ss.str();
}

template <typename T>
T StringToNumber ( const string_t &Text )//Text not by const reference so that the function can be used with a
{                               //character array as argument
   std::stringstream ss(Text);
   T result;
   return ss >> result ? result : 0;
}

class ItemProperty
{
private:
   int _valuei;
   float _valuef;
   string_t _values;

   int _originalType;
   static const int wasNull =0;
   static const int wasInt = 1;
   static const int wasFloat = 2;
   static const int wasString = 3;

public:
   ItemProperty()
   {
      *this = "null";
      _originalType = wasNull;
   }

   ItemProperty(int value)
   {
      *this = value;
   }
   
   ItemProperty(float value)
   {
      *this = value;
   }

   ItemProperty(string_t value)
   {
      *this = value;
   }

   ItemProperty(const ItemProperty &value)
   {
      *this = value;
   }

   virtual ~ItemProperty()
   {
      _valuei = 0;
      _valuef = 0.0f;
      _values = "";
   }

   operator int() { return _valuei; }
   operator float() { return _valuef; }
   operator string_t() {return _values; }
   operator const ItemProperty() {return *this;}

   ItemProperty& operator=(const ItemProperty &rhs)
   {
      if(this==&rhs) return *this;
      _valuei = rhs._valuei;
      _valuef = rhs._valuef;
      _values = rhs._values;
      _originalType = rhs._originalType;
      return *this;
   }
   
   ItemProperty& operator=(const int &rhs)
   {
      _valuei = rhs;
      _valuef = (float)rhs;
      _values = NumberToString(rhs);
      _originalType = wasInt;
      return *this;
   }

   ItemProperty& operator=(const float &rhs)
   {
      _valuei = (int)rhs;
      _valuef = rhs;
      _values = NumberToString(rhs);
      _originalType = wasFloat;
      return *this;
   }

   ItemProperty& operator=(const string_t &rhs)
   {
      _valuei = StringToNumber<int>(rhs);
      _valuef = StringToNumber<float>(rhs);
      _values = rhs;
      _originalType = wasString;
      return *this;
   }

   int operator+(const int &rhs)
   {
      return _valuei + rhs;
   }

   float operator+(const float &rhs)
   {
      return _valuef + rhs;
   }

   string_t operator+(const string_t &rhs)
   {
      return _values.append( rhs);
   }

   ItemProperty operator+(const ItemProperty &rhs)
   {
      switch((_originalType > rhs._originalType) ? _originalType : rhs._originalType)
      {
      case wasInt:
         return _valuei + rhs._valuei;
      case wasFloat:
         return _valuef + rhs._valuef;
      case wasString:
         return _values + rhs._values;
      default:
         return 0;
      }
   }

   int operator-(const int &rhs)
   {
      return _valuei - rhs;
   }

   float operator-(const float &rhs)
   {
      return _valuef - rhs;
   }

   ItemProperty operator-(const ItemProperty &rhs)
   {
      switch((_originalType > rhs._originalType) ? _originalType : rhs._originalType)
      {
      case wasInt:
         return _valuei - rhs._valuei;
      case wasFloat:
         return _valuef - rhs._valuef;
      default:
         return 0;
      }
   }

   int operator*(const int &rhs)
   {
      return _valuei * rhs;
   }

   float operator*(const float &rhs)
   {
      return _valuef * rhs;
   }

   ItemProperty operator*(const ItemProperty &rhs)
   {
      switch((_originalType > rhs._originalType) ? _originalType : rhs._originalType)
      {
      case wasInt:
         return _valuei * rhs._valuei;
      case wasFloat:
         return _valuef * rhs._valuef;
      default:
         return 0;
      }
   }

   int operator/(const int &rhs)
   {
      return _valuei / rhs;
   }

   float operator/(const float &rhs)
   {
      return _valuef / rhs;
   }
   
   ItemProperty operator/(const ItemProperty &rhs)
   {
      switch((_originalType > rhs._originalType) ? _originalType : rhs._originalType)
      {
      case wasInt:
         return _valuei / rhs._valuei;
      case wasFloat:
         return _valuef / rhs._valuef;
      default:
         return 0;
      }
   }

   ItemProperty& operator += (const ItemProperty &rhs)
   {
      *this = *this + rhs;
      return *this;
   }

   ItemProperty& operator += (const int &rhs)
   {
      *this = *this + rhs;
      return *this;
   }

   ItemProperty& operator += (const float &rhs)
   {
      *this = *this + rhs;
      return *this;
   }

   ItemProperty& operator += (const string_t &rhs)
   {
      *this = *this + rhs;
      return *this;
   }

   ItemProperty& operator -= (const ItemProperty &rhs)
   {
      *this = *this - rhs;
      return *this;
   }

   ItemProperty& operator -= (const int &rhs)
   {
      *this = *this - rhs;
      return *this;
   }

   ItemProperty& operator -= (const float &rhs)
   {
      *this = *this - rhs;
      return *this;
   }

   ItemProperty& operator *= (const ItemProperty &rhs)
   {
      *this = *this * rhs;
      return *this;
   }

   ItemProperty& operator *= (const int &rhs)
   {
      *this = *this * rhs;
      return *this;
   }

   ItemProperty& operator *= (const float &rhs)
   {
      *this = *this * rhs;
      return *this;
   }   

   ItemProperty& operator /= (const ItemProperty &rhs)
   {
      *this = *this / rhs;
      return *this;
   }

   ItemProperty& operator /= (const int &rhs)
   {
      *this = *this / rhs;
      return *this;
   }

   ItemProperty& operator /= (const float &rhs)
   {
      *this = *this / rhs;
      return *this;
   }
   
   //prefix
   ItemProperty& operator ++ ()
   {
      switch(_originalType)
      {
      case wasInt:
         *this = *this + 1;
         break;
      case wasFloat:
         *this = *this + 1.0f;
         break;
      default:
         break;
      }
      return *this;
   }

   //postfix
   ItemProperty& operator ++ (int dummy)
   {
      switch(_originalType)
      {
      case wasInt:
         *this = *this + 1;
         break;
      case wasFloat:
         *this = *this + 1.0f;
         break;
      default:
         break;
      }
      return *this;
   }

   //prefix
   ItemProperty& operator -- ()
   {
      switch(_originalType)
      {
      case wasInt:
         *this = *this - 1;
         break;
      case wasFloat:
         *this = *this - 1.0f;
         break;
      default:
         break;
      }
      return *this;
   }

   //postfix
   ItemProperty& operator -- (int dummy)
   {
      switch(_originalType)
      {
      case wasInt:
         *this = *this - 1;
         break;
      case wasFloat:
         *this = *this - 1.0f;
         break;
      default:
         break;
      }
      return *this;
   }
};


/*string_t operator +(const char * lhs, const ItemProperty& rhs)
{
   return string_t(lhs).append((const string_t)((ItemProperty&)rhs));
}*/

class Item
{
private:
   boost::unordered_map<string_t, ItemProperty> _properties;
public:
   Item()
   {
      
   }

   ItemProperty& operator[](const string_t& propertyName)
   {
      return _properties[propertyName];
   }

   bool has(string_t propertyName)
   {
      return _properties.find(propertyName) != _properties.end();
   }
};

#endif __ITEM_H__