#include "Math.h"
#include <cstring>


namespace basecode { namespace Math {

	vec3f::vec3f() : x(0.0f), y(0.0f), z(0.0f)
	{
	}

	vec3f::vec3f(float x0, float y0, float z0)
	{
		x = x0;
		y = y0;
		z = z0;
	}

	vec3f vec3f::cross(vec3f v)
	{
		vec3f p;

		p.x = y * v.z - v.y * z;
		p.y = z * v.x - v.z * x;
		p.z = x * v.y - v.x * y;

		return p;

	}

	float vec3f::dot(vec3f v)
	{
		return dot(v.x, v.y, v.z);
	}

	float vec3f::dot(float x0, float y0, float z0)
	{
		float ret;

		ret = x * x0 + y * y0 + z * z0;
		//ret /= magnitude() * sqrt(x0 * x0 + y0 * y0);

		return ret;
	}

	void vec3f::normalize()
	{
		float mag = magnitude();
		
		if (mag > 0)
		{
			x /= mag;
			y /= mag;
			z /= mag;
		}
	}

	float vec3f::magnitude()
	{
		if (x == 0 && y == 0 && z == 0) return 0;
		return sqrt(x * x + y * y + z * z);
	}

	float vec3f::magnitude2()
	{
		if (x == 0 && y == 0 && z == 0) return 0;
		return x * x + y * y + z * z;
	}

	vec3f vec3f::multiply(Matrix m)
	{
		vec3f p;

		float* d = m.data();

		p.x = x * d[0] + y * d[4] + z * d[8] + d[12];
		p.y = x * d[1] + y * d[5] + z * d[9] + d[13];
		p.z = x * d[2] + y * d[6] + z * d[10] + d[14];

		return p;
	}

	void vec3f::scale(float s0)
	{
		x *= s0;
		y *= s0;
		z *= s0;
	}

	void vec3f::set(float x0, float y0, float z0)
	{
		x = x0;
		y = y0;
		z = z0;
	}

	void vec3f::set(vec3f v)
	{
		set(v.x, v.y, v.z);
	}
	
	const char* vec3f::c_str()
	{
		std::stringstream ss;
		ss << "vec3f<" << x << ", " << y << ", " << z << ">";

		char* cstr = new char [ss.str().size()+1];
		strcpy (cstr, ss.str().c_str());
		
		return (const char*)cstr;
	}
	


	bool vec3f::operator==(const vec3f &rhs)
	{
		if (x == rhs.x && y == rhs.y && z == rhs.z) return true;
		return false;
	}

	bool vec3f::operator!=(const vec3f &rhs)
	{
		if (x != rhs.x || y != rhs.y || z != rhs.z) return true;
		return false;
	}

	vec3f& vec3f::operator+=(const vec3f &rhs)
	{
		x += rhs.x;
		y += rhs.y;
		z += rhs.z;

		return *this;
	 }


	vec3f& vec3f::operator-=(const vec3f &rhs)
	{
		x -= rhs.x;
		y -= rhs.y;
		z -= rhs.z;

		return *this;
	 }


	vec3f& vec3f::operator/=(const float &rhs)
	{
		x /= rhs;
		y /= rhs;
		z /= rhs;

		return *this;
	 }

	vec3f& vec3f::operator/=(const vec3f &rhs)
	{
		x /= rhs.x;
		y /= rhs.y;
		z /= rhs.z;

		return *this;
	 }

	vec3f& vec3f::operator*=(const float &rhs)
	{
		x *= rhs;
		y *= rhs;
		z *= rhs;

		return *this;
	 }

	vec3f& vec3f::operator*=(const vec3f &rhs)
	{
		x *= rhs.x;
		y *= rhs.y;
		z *= rhs.z;

		return *this;
	 }

	vec3f vec3f::operator+(const vec3f &rhs)
	{
		vec3f result = *this;
		result += rhs;
		return result;
		//*this = result;
		//return *this;
	 }


	vec3f vec3f::operator-(const vec3f &rhs)
	{
		vec3f result = *this;
		result -= rhs;
		return result;
		//*this = result;
		//return *this;
	 }

	vec3f vec3f::operator*(const float &rhs)
	{
		vec3f result = *this;
		result.x = result.x * rhs;
		result.y = result.y * rhs;
		result.z = result.z * rhs;

		return result;
		//*this = result;
		//return *this;
	 }

	vec3f vec3f::operator*(const vec3f &rhs)
	{
		vec3f result = *this;
		result.x = result.x * rhs.x;
		result.y = result.y * rhs.y;
		result.z = result.z * rhs.z;

		return result;
		//*this = result;
		//return *this;
	 }

	vec3f vec3f::operator/(const float &rhs)
	{
		vec3f result = *this;
		result.x = result.x / rhs;
		result.y = result.y / rhs;
		result.z = result.z / rhs;

		return result;
		//*this = result;
		//return *this;
	 }

	
	
	//-------------------------------------------------------------------------------------------------------
	vec2f::vec2f()
	{
		init(0, 0);
	}

	vec2f::vec2f(float x0, float y0)
	{
		init(x0, y0);
	}

	const char* vec2f::c_str()
	{
		std::stringstream ss;
		ss << "vec2f<" << x << ", " << y << ">";

		char* cstr = new char [ss.str().size()+1];
		strcpy (cstr, ss.str().c_str());
		
		return (const char*)cstr;
	}
	

	vec3f vec2f::cross(vec2f v)
	{
		vec3f p;

		p.x = y - v.y;
		p.y = v.x - x;
		p.z = x * v.y - v.x * y;

		return p;

	}

	void vec2f::init(float x0, float y0)
	{
		x = x0;
		y = y0;
	}

	float vec2f::dot(vec2f v)
	{
		return dot(v.x, v.y);
	}

	float vec2f::dot(float x0, float y0)
	{
		float ret;

		ret = x * x0 + y * y0;
		
		return ret;
	}

	void vec2f::intersection(vec2f p1, vec2f p2, vec2f p3, vec2f p4)
	{
		float	uA;

		uA =  (p1.x - p3.x) * (p4.y - p3.y) - (p1.y - p3.y) * (p4.x - p3.x);
		uA /= (p2.y - p1.y) * (p4.x - p3.x) - (p2.x - p1.x) * (p4.y - p3.y);

		x = p1.x + uA * (p2.x - p1.x);
		y = p1.y + uA * (p2.y - p1.y);

	}

	float vec2f::magnitude()
	{
		if (x == 0 && y == 0) return 0;
		return sqrt(x * x + y * y);
	}

	float vec2f::magnitude2()
	{
		if (x == 0 && y == 0) return 0;
		return x * x + y * y;
	}

	vec2f vec2f::multiply(Matrix m)
	{
		vec2f p;

		float* d = m.data();

		p.x = x * d[0] + y * d[4] + d[12];
		p.y = x * d[1] + y * d[5] + d[13];

		return p;
	}

	void vec2f::normalize()
	{
		float mag = magnitude();

		if (mag > 0)
		{
			x /= mag;
			y /= mag;
		}
	}

	void vec2f::scale(float s0)
	{
		x *= s0;
		y *= s0;
	}

	void vec2f::set(float x0, float y0)
	{
		x = x0;
		y = y0;
	}

	void vec2f::set(vec2f v0)
	{
		set(v0.x, v0.y);
	}

	bool vec2f::operator==(const vec2f &rhs)
	{
		if (x == rhs.x && y == rhs.y) return true;
		return false;
	}


	vec2f& vec2f::operator+=(const vec2f &rhs)
	{
		x += rhs.x;
		y += rhs.y;
		
		return *this;
	 }


	vec2f& vec2f::operator-=(const vec2f &rhs)
	{
		x -= rhs.x;
		y -= rhs.y;
		
		return *this;
	 }


	vec2f& vec2f::operator/=(const float &rhs)
	{
		x /= rhs;
		y /= rhs;
		
		return *this;
	 }

	vec2f& vec2f::operator/=(const vec2f &rhs)
	{
		x /= rhs.x;
		y /= rhs.y;
		
		return *this;
	 }

	vec2f& vec2f::operator*=(const float &rhs)
	{
		x *= rhs;
		y *= rhs;
		
		return *this;
	 }

	vec2f& vec2f::operator*=(const vec2f &rhs)
	{
		x *= rhs.x;
		y *= rhs.y;
		
		return *this;
	 }

	vec2f vec2f::operator+(const vec2f &rhs)
	{
		vec2f result = *this;
		result += rhs;
		return result;
		//*this = result;
		//return *this;
	 }


	vec2f vec2f::operator-(const vec2f &rhs)
	{
		vec2f result = *this;
		result -= rhs;
		return result;
		//*this = result;
		//return *this;
	 }

	vec2f vec2f::operator*(const float &rhs)
	{
		vec2f result = *this;
		result.x = result.x * rhs;
		result.y = result.y * rhs;
		
		return result;
		//*this = result;
		//return *this;
	 }

	vec2f vec2f::operator*(const vec2f &rhs)
	{
		vec2f result = *this;
		result.x = result.x * rhs.x;
		result.y = result.y * rhs.y;
		
		return result;
		//*this = result;
		//return *this;
	}

	vec2f vec2f::operator/(const float &rhs)
	{
		vec2f result = *this;
		result.x = result.x / rhs;
		result.y = result.y / rhs;
		
		return result;
		//*this = result;
		//return *this;
	 }

	vec2f vec2f::operator/(const vec2f &rhs)
	{
		vec2f result = *this;
		result.x = result.x / rhs.x;
		result.y = result.y / rhs.y;
		
		return result;
		//*this = result;
		//return *this;
	}



	//-------------------------------------------------------------------------------------------------------
	vec2i::vec2i()
	{
		init(0, 0);
	}

	vec2i::vec2i(int x0, int y0)
	{
		init(x0, y0);
	}

	vec2i::vec2i(float x0, float y0)
	{
		init((int)x0, (int)y0);
	}

	void vec2i::init(int x0, int y0)
	{
		x = x0;
		y = y0;
	}

	float vec2i::magnitude()
	{
		if (x == 0 && y == 0) return 0;
		return sqrt(float(x * x) + float(y * y));
	}

	vec2f vec2i::multiply(Matrix m)
	{
		vec2f p;

		float* d = m.data();

		p.x = (float)x * d[0] + (float)y * d[4] + d[12];
		p.y = (float)x * d[1] + (float)y * d[5] + d[13];

		return p;
	}

	void vec2i::scale(int s0)
	{
		x *= s0;
		y *= s0;
	}

	void vec2i::set(int x0, int y0)
	{
		x = x0;
		y = y0;
	}

	void vec2i::set(vec2i v)
	{
		set(v.x, v.y);
	}

	const char* vec2i::c_str()
	{
		std::stringstream ss;
		ss << "vec2i<" << x << ", " << y << ">";

		char* cstr = new char [ss.str().size()+1];
		strcpy (cstr, ss.str().c_str());
		
		return (const char*)cstr;
	}


	bool vec2i::operator==(const vec2i &rhs)
	{
		if (x == rhs.x && y == rhs.y) return true;
		return false;
	}


	vec2i& vec2i::operator+=(const vec2i &rhs)
	{
		x += rhs.x;
		y += rhs.y;
		
		return *this;
	 }


	vec2i& vec2i::operator-=(const vec2i &rhs)
	{
		x -= rhs.x;
		y -= rhs.y;
		
		return *this;
	 }


	vec2i& vec2i::operator/=(const float &rhs)
	{
		x = int(x / rhs);
		y = int(y / rhs);
		
		return *this;
	 }

	vec2i& vec2i::operator*=(const float &rhs)
	{
		x = int(x * rhs);
		y = int(y * rhs);
		
		return *this;
	 }

	vec2i vec2i::operator+(const vec2i &rhs)
	{
		vec2i result = *this;
		result += rhs;
		return result;
		//*this = result;
		//return *this;
	 }


	vec2i vec2i::operator-(const vec2i &rhs)
	{
		vec2i result = *this;
		result -= rhs;
		return result;
		//*this = result;
		//return *this;
	 }

	vec2i vec2i::operator*(const float &rhs)
	{
		vec2i result = *this;
		result.x = int(result.x * rhs);
		result.y = int(result.y * rhs);
		
		return result;
		//*this = result;
		//return *this;
	 }

	vec2i vec2i::operator*(const vec2i &rhs)
	{
		vec2i result = *this;
		result.x = result.x * rhs.x;
		result.y = result.y * rhs.y;
		
		return result;
		//*this = result;
		//return *this;
	 }

	//-------------------------------------------------------------------------------------------------------
	vec4i::vec4i(int x0, int y0, int z0, int w0)
	{
		init(x0, y0, z0, w0);
	}

	vec4i::vec4i(float x0, float y0, float z0, float w0)
	{
		init(int(x0), int(y0), int(z0), int(w0));
	}

	vec4i::vec4i()
	{
		init(0, 0, 0, 0);
	}

	
	void vec4i::init(int x0, int y0, int z0, int w0)
	{
		x = x0;
		y = y0;
		z = z0;
		w = w0;
	}

	void vec4i::scale(int s0)
	{
		x *= s0;
		y *= s0;
		z *= s0;
		w *= s0;
	}

	void vec4i::set(int x0, int y0, int z0, int w0)
	{
		x = x0;
		y = y0;
		z = z0;
		w = w0;
	}

	void vec4i::set(float x0, float y0, float z0, float w0)
	{
		set(int(x0), int(y0), int(z0), int(w0));
	}

	void vec4i::set(vec4i v0)
	{
		set(v0.x, v0.y, v0.z, v0.w);
	}

	const char* vec4i::c_str()
	{
		std::stringstream ss;
		ss << "vec4i<" << x << ", " << y << ", " << z << ", " << w << ">";

		char* cstr = new char [ss.str().size()+1];
		strcpy (cstr, ss.str().c_str());
		
		return (const char*)cstr;
	}

	
	//-------------------------------------------------------------------------------------------------------

	Matrix::Matrix()
	{
		loadIdentity();
	}

	float* Matrix::data()
	{
		return &m[0];
	}

	void Matrix::glMultMatrixf()
	{
		::glMultMatrixf(data());
	}

	void Matrix::glLoadMatrixf()
	{
		::glLoadMatrixf(data());
	}

	void Matrix::loadIdentity()
	{
		for (int i = 0; i < 16; i++) m[i] = 0;

		m[0] = 1;
		m[5] = 1;
		m[10] = 1;
		m[15] = 1;
	}

	void Matrix::multiply(Matrix t)
	{
		float tmp[16];
		float* tm = t.data();

		tmp[0] = (tm[0] * m[0]) + (tm[1] * m[4]) + (tm[2] * m[8]) + (tm[3] * m[12]);
		tmp[1] = (tm[0] * m[1]) + (tm[1] * m[5]) + (tm[2] * m[9]) + (tm[3] * m[13]);
		tmp[2] = (tm[0] * m[2]) + (tm[1] * m[6]) + (tm[2] * m[10]) + (tm[3] * m[14]);
		tmp[3] = (tm[0] * m[3]) + (tm[1] * m[7]) + (tm[2] * m[11]) + (tm[3] * m[15]);

		tmp[4] = (tm[4] * m[0]) + (tm[5] * m[4]) + (tm[6] * m[8]) + (tm[7] * m[12]);
		tmp[5] = (tm[4] * m[1]) + (tm[5] * m[5]) + (tm[6] * m[9]) + (tm[7] * m[13]);
		tmp[6] = (tm[4] * m[2]) + (tm[5] * m[6]) + (tm[6] * m[10]) + (tm[7] * m[14]);
		tmp[7] = (tm[4] * m[3]) + (tm[5] * m[7]) + (tm[6] * m[11]) + (tm[7] * m[15]);

		tmp[8] = (tm[8] * m[0]) + (tm[9] * m[4]) + (tm[10] * m[8]) + (tm[11] * m[12]);
		tmp[9] = (tm[8] * m[1]) + (tm[9] * m[5]) + (tm[10] * m[9]) + (tm[11] * m[13]);
		tmp[10] = (tm[8] * m[2]) + (tm[9] * m[6]) + (tm[10] * m[10]) + (tm[11] * m[14]);
		tmp[11] = (tm[8] * m[3]) + (tm[9] * m[7]) + (tm[10] * m[11]) + (tm[11] * m[15]);

		tmp[12] = (tm[12] * m[0]) + (tm[13] * m[4]) + (tm[14] * m[8]) + (tm[15] * m[12]);
		tmp[13] = (tm[12] * m[1]) + (tm[13] * m[5]) + (tm[14] * m[9]) + (tm[15] * m[13]);
		tmp[14] = (tm[12] * m[2]) + (tm[13] * m[6]) + (tm[14] * m[10]) + (tm[15] * m[14]);
		tmp[15] = (tm[12] * m[3]) + (tm[13] * m[7]) + (tm[14] * m[11]) + (tm[15] * m[15]);

		memcpy(&m[0], &tmp[0], 16 * sizeof(float));
	}

	void Matrix::orient(Matrix t)
	{
		float* d = t.data();
		for (int i = 0; i < 3; i++)
		{
			m[0 + i] = d[0 + i];
			m[4 + i] = d[4 + i];
			m[8 + i] = d[8 + i];
		}
	}

	void Matrix::orient(vec3f vx, vec3f vy, vec3f vz)
	{
		m[0] = vx.x;
		m[1] = vx.y;
		m[2] = vx.z;
		m[4] = vy.x;
		m[5] = vy.y;
		m[6] = vy.z;
		m[8] = vz.x;
		m[9] = vz.y;
		m[10] = vz.z;
	}

	void Matrix::position(vec3f v)
	{
		position(v.x, v.y, v.z);
	}

	void Matrix::position(float x0, float y0, float z0)
	{
		m[12] = x0;
		m[13] = y0;
		m[14] = z0;
	}

	void Matrix::rotate(float an0)
	{
		Matrix tm;
		float tcos, tsin;

		tm.loadIdentity();
		tcos = cos(an0 * 3.14159f / 180);
		tsin = sin(an0 * 3.14159f / 180);
		tm.m[0] = tcos;
		tm.m[1] = tsin;
		tm.m[4] = -tsin;
		tm.m[5] = tcos;
		multiply(tm);
	
	}

	void Matrix::rotate(vec3f v)
	{
		rotate(v.x, v.y, v.z);
	}

	void Matrix::rotate(float anX, float anY, float anZ)
	{
		Matrix tm;
		float tcos, tsin;

		if (anX != 0)
		{
			tm.loadIdentity();
			tcos = cos(anX * 3.14159f / 180.0f);
			tsin = sin(anX * 3.14159f / 180.0f);
			tm.m[5] = tcos;
			tm.m[6] = tsin;
			tm.m[9] = -tsin;
			tm.m[10] = tcos;
			multiply(tm);
		}

		if (anY != 0)
		{
			tm.loadIdentity();
			tcos = cos(anY * 3.14159f / 180.0f);
			tsin = sin(anY * 3.14159f / 180.0f);
			tm.m[0] = tcos;
			tm.m[2] = -tsin;
			tm.m[8] = tsin;
			tm.m[10] = tcos;
			multiply(tm);
		}

		if (anZ != 0)
		{
			tm.loadIdentity();
			tcos = cos(anZ * 3.14159f / 180.0f);
			tsin = sin(anZ * 3.14159f / 180.0f);
			tm.m[0] = tcos;
			tm.m[1] = tsin;
			tm.m[4] = -tsin;
			tm.m[5] = tcos;
			multiply(tm);
		}
	}


	void Matrix::translate(vec3f v)
	{
		translate(v.x, v.y, v.z);
	}

	void Matrix::translate(float x0, float y0, float z0)
	{
		Matrix t;
		t.loadIdentity();
		t.position(x0, y0, z0);

		multiply(t);	
	}

	void Matrix::transpose()
	{
		Matrix t;
		
		for (int x = 0; x < 4; x++)
			for (int y = 0; y < 4; y++)
				t.m[y * 4 + x] = m[x * 4 + y];

		for (int i = 0; i < 16; i++)
			m[i] = t.m[i];

	}

	vec3f Matrix::x()
	{
		return vec3f(m[0], m[1], m[2]);
	}

	vec3f Matrix::y()
	{
		return vec3f(m[4], m[5], m[6]);
	}

	vec3f Matrix::z()
	{
		return vec3f(m[8], m[9], m[10]);
	}
	

	//-------------------------------------------------------------------------------------------------------
	rectf::rectf(int l, int t, int r, int b)
	{
		init((float)l, (float)t, (float)r, (float)b);
	}

	rectf::rectf(float l, float t, float r, float b)
	{
		init(l, t, r, b);
	}

	rectf::rectf()
	{
		init(0, 0, 0, 0);
	}

	float rectf::area()
	{
		return (right - left) * (bottom - top);
	}

	vec2f rectf::center()
	{
		vec2f v0 = vec2f((left + right) / 2.0f, (top + bottom) / 2.0f);
		return v0;
	}

	
	void rectf::init(float l, float t, float r, float b)
	{
		top = t;
		bottom = b;
		left = l;
		right = r;
	}

	void rectf::forceInside(float l, float t, float r, float b)
	{
		float dx, dy;

		if (left < l)
		{
			dx = l - left;
			left += dx;
			right += dx;
			if (right > r) right = r;
		}

		if (right > r)
		{
			dx = r - right;
			left += dx;
			right += dx;
			if (left < l) left = l;
		}
		
		if (top < t)
		{
			dy = t - top;
			top += dy;
			bottom += dy;
			if (bottom > b) bottom = b;
		}

		if (bottom > b)
		{
			dy = b - bottom;
			top += dy;
			bottom += dy;
			if (top < t) top = t;
		}
	}

	void rectf::forceInside(int l, int t, int r, int b)
	{
		forceInside(float(l), float(t), float(r), float(b));
	}

	void rectf::forceInside(rectf r)
	{
		forceInside(r.left, r.top, r.right, r.bottom);
	}

	bool rectf::isInside(int l, int t, int r, int b)
	{
		return isInside(float(l), float(t), float(r), float(b));
	}
	
	bool rectf::isInside(float l, float t, float r, float b)
	{
		// is parameter inside this?
		if (l >= left && t >= top && r <= right && b <= bottom) return true;
		return false;
	}

	bool rectf::isInside(rectf r0)
	{
		return isInside(r0.left, r0.top, r0.right, r0.bottom);
	}
	
	bool rectf::isInside(recti r0)
	{
		return isInside(r0.left, r0.top, r0.right, r0.bottom);
	}
	
	bool rectf::isInside(vec2i v0)
	{
		return isInside(float(v0.x), float(v0.y), float(v0.x), float(v0.y));
	}
	
	bool rectf::isInside(vec2f v0)
	{
		return isInside(v0.x, v0.y, v0.x, v0.y);
	}

	bool rectf::isInside(float x0, float y0)
	{
		return isInside(x0, y0, x0, y0);
	}

	bool rectf::isInside(int x0, int y0)
	{
		return isInside(x0, y0, x0, y0);
	}

	////

	bool rectf::isInside_explicit(int l, int t, int r, int b)
	{
		return isInside_explicit(float(l), float(t), float(r), float(b));
	}
	
	bool rectf::isInside_explicit(float l, float t, float r, float b)
	{
		// is parameter inside this?
		if (l > left && t > top && r < right && b < bottom) return true;
		return false;
	}

	bool rectf::isInside_explicit(rectf r0)
	{
		return isInside_explicit(r0.left, r0.top, r0.right, r0.bottom);
	}
	
	bool rectf::isInside_explicit(recti r0)
	{
		return isInside_explicit(r0.left, r0.top, r0.right, r0.bottom);
	}
	
	bool rectf::isInside_explicit(vec2i v0)
	{
		return isInside_explicit(float(v0.x), float(v0.y), float(v0.x), float(v0.y));
	}
	
	bool rectf::isInside_explicit(vec2f v0)
	{
		return isInside_explicit(v0.x, v0.y, v0.x, v0.y);
	}

	bool rectf::isInside_explicit(float x0, float y0)
	{
		return isInside_explicit(x0, y0, x0, y0);
	}

	bool rectf::isInside_explicit(int x0, int y0)
	{
		return isInside_explicit(x0, y0, x0, y0);
	}

	////

	bool rectf::overlaps(rectf r0)
	{
		if (isInside(r0.left, r0.top)) return true;
		if (isInside(r0.left, r0.bottom)) return true;
		if (isInside(r0.right, r0.top)) return true;
		if (isInside(r0.right, r0.bottom)) return true;

		if (r0.isInside(left, top)) return true;
		if (r0.isInside(left, bottom)) return true;
		if (r0.isInside(right, top)) return true;
		if (r0.isInside(right, bottom)) return true;

		return false;
	}

	bool rectf::overlaps(recti r0)
	{
		if (isInside(r0.left, r0.top)) return true;
		if (isInside(r0.left, r0.bottom)) return true;
		if (isInside(r0.right, r0.top)) return true;
		if (isInside(r0.right, r0.bottom)) return true;

		if (r0.isInside(left, top)) return true;
		if (r0.isInside(left, bottom)) return true;
		if (r0.isInside(right, top)) return true;
		if (r0.isInside(right, bottom)) return true;

		return false;
	}

	void rectf::scale(float s0)
	{
		top *= s0;
		bottom *= s0;
		left *= s0;
		right *= s0;
	}

	void rectf::set(float l, float t, float r, float b)
	{
		top = t;
		bottom = b;
		left = l;
		right = r;
	}

	void rectf::set(int l, int t, int r, int b)
	{
		set(float(l), float(t), float(r), float(b));
	}

	void rectf::set(rectf r0)
	{
		set(r0.left, r0.top, r0.right, r0.bottom);
	}

	const char* rectf::c_str()
	{
		std::stringstream ss;
		ss << "rectf<" << left << ", " << top << ", " << right << ", " << bottom << ">";

		char* cstr = new char [ss.str().size()+1];
		strcpy (cstr, ss.str().c_str());
		
		return (const char*)cstr;
	}
	
	rectf& rectf::operator/=(const float &rhs)
	{
		left /= rhs;
		right /= rhs;
		top /= rhs;
		bottom /= rhs;

		return *this;
	 }

	rectf& rectf::operator*=(const float &rhs)
	{
		left *= rhs;
		right *= rhs;
		top *= rhs;
		bottom *= rhs;

		return *this;
	 }

	//-------------------------------------------------------------------------------------------------------
	recti::recti(int l, int t, int r, int b)
	{
		init(l, t, r, b);
	}

	recti::recti(float l, float t, float r, float b)
	{
		init(int(l), int(t), int(r), int(b));
	}

	recti::recti()
	{
		init(0, 0, 0, 0);
	}

	int recti::area()
	{
		return (right - left) * (bottom - top);
	}
	
	vec2i recti::center()
	{
		vec2i v0 = vec2i((left + right) / 2, (top + bottom) / 2);
		return v0;
	}

	
	void recti::init(int l, int t, int r, int b)
	{
		top = t;
		bottom = b;
		left = l;
		right = r;
	}

	void recti::forceInside(int l, int t, int r, int b)
	{
		int dx, dy;

		if (left < l)
		{
			dx = l - left;
			left += dx;
			right += dx;
			if (right > r) right = r;
		}

		if (right > r)
		{
			dx = r - right;
			left += dx;
			right += dx;
			if (left < l) left = l;
		}
		
		if (top < t)
		{
			dy = t - top;
			top += dy;
			bottom += dy;
			if (bottom > b) bottom = b;
		}

		if (bottom > b)
		{
			dy = b - bottom;
			top += dy;
			bottom += dy;
			if (top < t) top = t;
		}
	}

	void recti::forceInside(float l, float t, float r, float b)
	{
		forceInside(int(l), int(t), int(r), int(b));
	}

	void recti::forceInside(recti r)
	{
		forceInside(r.left, r.top, r.right, r.bottom);
	}

	bool recti::isInside(int l, int t, int r, int b)
	{
		if (l >= left && t >= top && r <= right && b <= bottom) return true;
		return false;
	}
	
	bool recti::isInside(float l, float t, float r, float b)
	{
		return isInside(int(l), int(t), int(r), int(b));
	}
	
	bool recti::isInside(rectf r0)
	{
		return isInside(r0.left, r0.top, r0.right, r0.bottom);
	}
	
	bool recti::isInside(recti r0)
	{
		return isInside(r0.left, r0.top, r0.right, r0.bottom);
	}
	
	bool recti::isInside(vec2i v0)
	{
		return isInside(v0.x, v0.y, v0.x, v0.y);
	}
	
	bool recti::isInside(vec2f v0)
	{
		return isInside(int(v0.x), int(v0.y), int(v0.x), int(v0.y));
	}

	bool recti::isInside(float x0, float y0)
	{
		return isInside(x0, y0, x0, y0);
	}

	bool recti::isInside(int x0, int y0)
	{
		return isInside(x0, y0, x0, y0);
	}

	////

	bool recti::isInside_explicit(int l, int t, int r, int b)
	{
		return isInside_explicit(float(l), float(t), float(r), float(b));
	}
	
	bool recti::isInside_explicit(float l, float t, float r, float b)
	{
		// is parameter inside this?
		if (l > left && t > top && r < right && b < bottom) return true;
		return false;
	}

	bool recti::isInside_explicit(rectf r0)
	{
		return isInside_explicit(r0.left, r0.top, r0.right, r0.bottom);
	}
	
	bool recti::isInside_explicit(recti r0)
	{
		return isInside_explicit(r0.left, r0.top, r0.right, r0.bottom);
	}
	
	bool recti::isInside_explicit(vec2i v0)
	{
		return isInside_explicit(float(v0.x), float(v0.y), float(v0.x), float(v0.y));
	}
	
	bool recti::isInside_explicit(vec2f v0)
	{
		return isInside_explicit(v0.x, v0.y, v0.x, v0.y);
	}

	bool recti::isInside_explicit(float x0, float y0)
	{
		return isInside_explicit(x0, y0, x0, y0);
	}

	bool recti::isInside_explicit(int x0, int y0)
	{
		return isInside_explicit(x0, y0, x0, y0);
	}

	////
	
	bool recti::overlaps(rectf r0)
	{
		if (isInside(r0.left, r0.top)) return true;
		if (isInside(r0.left, r0.bottom)) return true;
		if (isInside(r0.right, r0.top)) return true;
		if (isInside(r0.right, r0.bottom)) return true;

		if (r0.isInside(left, top)) return true;
		if (r0.isInside(left, bottom)) return true;
		if (r0.isInside(right, top)) return true;
		if (r0.isInside(right, bottom)) return true;

		return false;
	}

	bool recti::overlaps(recti r0)
	{
		if (isInside(r0.left, r0.top)) return true;
		if (isInside(r0.left, r0.bottom)) return true;
		if (isInside(r0.right, r0.top)) return true;
		if (isInside(r0.right, r0.bottom)) return true;

		if (r0.isInside(left, top)) return true;
		if (r0.isInside(left, bottom)) return true;
		if (r0.isInside(right, top)) return true;
		if (r0.isInside(right, bottom)) return true;

		return false;
	}

	void recti::scale(int s0)
	{
		top *= s0;
		bottom *= s0;
		left *= s0;
		right *= s0;
	}

	void recti::set(int l, int t, int r, int b)
	{
		top = t;
		bottom = b;
		left = l;
		right = r;
	}

	void recti::set(float l, float t, float r, float b)
	{
		set(int(l), int(t), int(r), int(b));
	}

	void recti::set(recti r0)
	{
		set(r0.left, r0.top, r0.right, r0.bottom);
	}

	const char* recti::c_str()
	{
		std::stringstream ss;
		ss << "recti<" << left << ", " << top << ", " << right << ", " << bottom << ">";

		char* cstr = new char [ss.str().size()+1];
		strcpy (cstr, ss.str().c_str());
		
		return (const char*)cstr;
	}

	//------------------------------------------------------------------------------------------------------
	AxisAngle::AxisAngle()
	{
		axis.set(0, 0, 0);
		angle = 0;

		calcMatrix();
	}


	AxisAngle::AxisAngle(float angle0, float x0, float y0, float z0)
	{
		set(angle0, x0, y0, z0);
	}

	AxisAngle::AxisAngle(float angle0, vec3f axis0)
	{
		set(angle0, axis0);
	}

	const char* AxisAngle::c_str()
	{
		std::stringstream ss;
		ss << "AxisAngle<" << angle << ", " << axis.x << ", " << axis.y << ", " << axis.z << ">";

		char* cstr = new char [ss.str().size()+1];
		strcpy (cstr, ss.str().c_str());
		
		return (const char*)cstr;
	}
	
	void AxisAngle::calcMatrix()
	{
		float ux = axis.x;
		float uy = axis.y;
		float uz = axis.z;
		float ux2 = ux * ux;
		float uy2 = uy * uy;
		float uz2 = uz * uz;
		float c = cos(angle);
		float s = sin(angle);

		matrix.loadIdentity();

/*		t*x*x + c  	t*x*y - z*s  	t*x*z + y*s
t*x*y + z*s 	t*y*y + c 	t*y*z - x*s
t*x*z - y*s 	t*y*z + x*s 	t*z*z + c

    *  c =cos(angle)
    * s = sin(angle)
    * t =1 - c
    * x = normalised axis x coordinate
    * y = normalised axis y coordinate
    * z = normalised axis z coordinate
*/

		float* m = matrix.data();
		/*m[0] = (1 - c) * ux2 + c;
		m[1] = (1 - c) * ux * uy - uz * s;
		m[2] = (1 - c) * ux * uz + uy * s;

		m[4] = (1 - c) * ux * uy + uz * s;
		m[5] = (1 - c) * uy2 + c;
		m[6] = (1 - c) * uy * uz - ux * s;

		m[8] = (1 - c) * ux * uz - uy * s;
		m[9] = (1 - c) * uy * uz + ux * s;
		m[10] = (1 - c) * uz2 + c;*/

		m[0] = ux2 + (1 - ux2) * c;
		m[1] = ux * uy * (1 - c) - uz * s;
		m[2] = ux * uz * (1 - c) + uy * s;
		
		m[4] = ux * uy * (1 - c) + uz * s;
		m[5] = uy2 + (1 - uy2) * c;
		m[6] = uy * uz * (1 - c) - ux * s;

		m[8] = ux * uz * (1 - c) - uy * s;
		m[9] = uy * uz * (1 - c) + ux * s;
		m[10] = uz2 + (1 - uz2) * c;
	}

	void AxisAngle::normalize()
	{
		axis.normalize();
	}

	void AxisAngle::rotate(float angle0)
	{
		angle += angle0;

		calcMatrix();
	}

	/*void AxisAngle::rotate(vec3f angle0)
	{
		Matrix m;
		float* d = m.data();

		//for (int i = 0; i < 16; i++) d[i] = 0;

		d[1] = -angle0.z;
		d[2] = angle0.y;
		d[4] = angle0.z;
		d[6] = -angle0.x;
		d[8] = -angle0.y;
		d[9] = angle0.x;

		matrix.multiply(m);
	}*/

	void AxisAngle::set(float angle0, float x0, float y0, float z0)
	{
		axis.set(x0, y0, z0);
		axis.normalize();
		angle = angle0;

		calcMatrix();
	}

	void AxisAngle::set(float angle0, vec3f axis0)
	{
		set(angle0, axis0.x, axis0.y, axis0.z);
	}

	void AxisAngle::set(float angle0)
	{
		angle = angle0;
		calcMatrix();
	}
	
	void AxisAngle::set(vec3f axis0)
	{
		axis = axis0;
		axis.normalize();

		calcMatrix();
	}

	//------------------------------------------------------------------------------------------------------
	quat4f::quat4f()
	{
		w = 0;
		x = 0;
		y = 0;
		z = 0;
	}

	quat4f::quat4f(float w0, float x0, float y0, float z0)
	{
		set(w0, x0, y0, z0);
	}

	quat4f::quat4f(float w0, vec3f v0)
	{
		set(w0, v0);
	}
	
	quat4f::quat4f(AxisAngle a0)
	{
		set(a0);
	}
	
	const char* quat4f::c_str()
	{
		std::stringstream ss;
		ss << "quat4f<" << w << ", " << x << ", " << y << ", " << z << ">";

		char* cstr = new char [ss.str().size()+1];
		strcpy (cstr, ss.str().c_str());
		
		return (const char*)cstr;
	}
	
	vec3f quat4f::axis()
	{
		vec3f n;
		n.set(x, y, z);
		n.normalize();

		return n;
	}

	void quat4f::calcMatrix()
	{
		float* d = matrix.data();

		float wx = w * x;
		float xy = x * y;
		float xz = x * z;		
		float wz = w * z;
		float yz = y * z;
		float wy = w * y;

		float w2 = w * w;
		float x2 = x * x;
		float y2 = y * y;
		float z2 = z * z;

		//d[0] = 1 - 2 * y2 - 2 * z2;
		d[0] = w2 + x2 - y2 - z2;
		d[1] = 2 * xy - 2 * wz;
		d[2] = 2 * xz + 2 * wy;
		d[3] = 0;

		d[4] = 2 * xy + 2 * wz;
		//d[5] = 1 - 2 * x2 - 2 * z2;
		d[5] = w2 - x2 + y2 - z2;
		d[6] = 2 * yz - 2 * wx;
		d[7] = 0;
		
		d[8] = 2 * xz - 2 * wy;
		d[9] = 2 * yz + 2 * wx;
		//d[10] = 1 - 2 * x2 - 2 * y2;
		d[10] = w2 - x2 - y2 + z2;
		d[11] = 0;

		d[12] = 0;
		d[13] = 0;
		d[14] = 0;
		d[15] = 1;

	}

	vec3f quat4f::euler()
	{
		vec3f v0;

		float x2 = x * x;
		float y2 = y * y;
		float z2 = z * z;

		v0.x = atan2(2 * (w * x + y * z), 1 - 2 * (x2 + y2));
		v0.y = asin(2 * (w * y - z * x));
		v0.z = atan2(2 * (w * z + x * y), 1 - 2 * (y2 + z2));

		return v0;
	}

	float quat4f::magnitude()
	{
		if (w == 0 && x == 0 && y == 0 && z == 0) return 0;
		return sqrt(w * w + x * x + y * y + z * z);
	}

	quat4f quat4f::multiply(quat4f rhs)
	{
		quat4f result;

		result.w = w * rhs.w - x * rhs.x - y * rhs.y - z * rhs.z;
		result.x = w * rhs.x + x * rhs.w + y * rhs.z - z * rhs.y;
		result.y = w * rhs.y - x * rhs.z + y * rhs.w + z * rhs.x;
		result.z = w * rhs.z + x * rhs.y - y * rhs.x + z * rhs.w;

		return result;
	}
	
	quat4f quat4f::multiply(float rhs)
	{
		quat4f result = *this;

		result.w *= rhs;
		result.x *= rhs;
		result.y *= rhs;
		result.z *= rhs;

		return result;
	}
	
	void quat4f::normalize()
	{
		float mag = magnitude();

		if (mag > 0)
		{
			w /= mag;
			x /= mag;
			y /= mag;
			z /= mag;
		}
	}

	void quat4f::set(float w0, float x0, float y0, float z0)
	{
		w = w0;
		x = x0;
		y = y0;
		z = z0;
	}

	void quat4f::set(float w0, vec3f v0)
	{
		set(w0, v0.x, v0.y, v0.z);
	}

	void quat4f::set(AxisAngle a0)
	{
		float c = cos(a0.angle / 2);
		float s = sin(a0.angle / 2);

		w = c;
		x = a0.axis.x * s;
		y = a0.axis.y * s;
		z = a0.axis.z * s;
	}

	void quat4f::set(quat4f q0)
	{
		set(q0.w, q0.x, q0.y, q0.z);
	}

	quat4f& quat4f::operator+=(const quat4f &rhs)
	{
		w += rhs.w;
		x += rhs.x;
		y += rhs.y;
		z += rhs.z;

		return *this;
	 }

	quat4f& quat4f::operator+(const quat4f &rhs)
	{
		w += rhs.w;
		x += rhs.x;
		y += rhs.y;
		z += rhs.z;

		return *this;
	 }

	quat4f& quat4f::operator*=(const quat4f &rhs)
	{
		quat4f q = *this;
		quat4f result = q.multiply(rhs);
		
		w = result.w;
		x = result.x;
		y = result.y;
		z = result.z;

		return *this;
	 }

	quat4f quat4f::operator*(const quat4f &rhs)
	{
		quat4f q = *this;
		quat4f result = q.multiply(rhs);
		
		return result;
		//*this = result;
		//return *this;
	 }


	quat4f quat4f::operator*(const float &rhs)
	{
		quat4f q = *this;
		quat4f result = q.multiply(rhs);
		
		return result;
		//*this = result;
		//return *this;
	 }

	//------------------------------------------------------------------------------------------------------

	Diffeq3f::Diffeq3f()
	{
		init();
	}

	void Diffeq3f::init()
	{
		clearSprings();
		//forces.init();
		//position.init();
		//velocity.init();
		//g.init();
		mass.set(1, 1, 1);
	}

	void Diffeq3f::addForce(vec3f f)
	{
		forces += f;
	}

	void Diffeq3f::addSpring(float ck, vec3f target)
	{
		k_values.push_back(ck);
		k_targets.push_back(target);
	}

	void Diffeq3f::clearForces()
	{
		forces.set();
	}

	void Diffeq3f::clearSprings()
	{
		k_values.clear();
		k_targets.clear();
	}
		
	void Diffeq3f::setDampening(float d)
	{
		g.set(d, d, d);
	}

	void Diffeq3f::setDampening(vec3f d)
	{
		g = d;
	}

	void Diffeq3f::setForces(vec3f f)
	{
		forces = f;
	}

	void Diffeq3f::setMass(float m)
	{
		mass.set(m, m, m);
	}

	void Diffeq3f::setMass(vec3f m)
	{
		mass = m;
	}

	void Diffeq3f::setPosition(vec3f s)
	{
		position = s;
	}

	void Diffeq3f::setVelocity(vec3f v)
	{
		velocity = v;
	}

	void Diffeq3f::solve(float dt)
	{
		position += velocity * dt;

		vec3f a;
		a = velocity * g * -1 + forces;

		for (unsigned int s = 0; s < k_values.size(); s++)
		{
			a -= (position - k_targets[s]) * k_values[s]; 
		}

		a /= mass;
		
		velocity += a * dt;
	}

	void DESolve(std::vector<Diffeq3f> &stack, float dt)
	{
		for(unsigned int i = 0; i < stack.size(); i++)
		{
			stack[i].solve(dt);
		}
	}


	//------------------------------------------------------------------------------------------------------
	float rnd()
	{
		float f = rand() % 1000 / (float)1000;

		return f;
	}

	float nrnd()
	{
		float f = ((rand() % 1000) - 500) / (float)500;

		return f;
	}

	float sgn(float f0)
	{
		if (f0 < 0) return -1;
		if (f0 == 0) return 0;
		return 1;
	}

	
}}