#ifndef __BASECODE_MATH_H__
#define __BASECODE_MATH_H__

#include <math.h>
#include <string>
#include <sstream>
#include <vector>
#include <time.h>
#include <stdio.h>
#include <stdlib.h>
#include <GL/glew.h>
#define GLFW_DLL
#include <GL/glfw.h>
#include <boost/archive/text_oarchive.hpp>

namespace basecode { namespace Math {

	class vec3f;
	class vec2f;
	class vec2i;
	class vec4i;
	class recti;
	class rectf;
	class Matrix;
	class AxisAngle;
	class quat4f;
	class Diffeq3f;

	const float PI = 3.141592653589793238f;

	class vec3f
	{
		friend class boost::serialization::access;
		template<class Archive>
		void serialize(Archive & ar, const unsigned int version)
		{
			ar & x;
			ar & y;
			ar & z;
		}
	public:
		float x, y, z;

		vec3f();
		vec3f(float x0, float y0, float z0);
		vec3f	cross(vec3f v);
		float	dot(vec3f v);
		float	dot(float x0, float y0, float z0);
		void	normalize();
		float	magnitude();
		float	magnitude2();
		vec3f	multiply(Matrix m);
		void	scale(float s0);
		void	set(float x0 = 0.0f, float y0 = 0.0f, float z0 = 0.0f);
		void	set(vec3f v);

		const char* c_str();
		
		bool operator==(const vec3f &rhs);
		bool operator!=(const vec3f &rhs);
		vec3f& operator+=(const vec3f &rhs);
		vec3f& operator-=(const vec3f &rhs);
		vec3f& operator/=(const float &rhs);
		vec3f& operator/=(const vec3f &rhs);
		vec3f& operator*=(const float &rhs);
		vec3f& operator*=(const vec3f &rhs);
		vec3f operator+(const vec3f &rhs);
		vec3f operator-(const vec3f &rhs);		
		vec3f operator*(const float &rhs);
		vec3f operator*(const vec3f &rhs);
		vec3f operator/(const float &rhs);
		

	};

	class vec2f
	{
		friend class boost::serialization::access;
		template<class Archive>
		void serialize(Archive & ar, const unsigned int version)
		{
			ar & x;
			ar & y;
		}
	public:
		float x, y;

		vec2f();
		vec2f(float x0, float y0);
		vec3f	cross(vec2f v);
		float	dot(vec2f v);
		float	dot(float x0, float y0);
		void	intersection(vec2f p1, vec2f p2, vec2f p3, vec2f p4);
		float	magnitude();
		float	magnitude2();
		vec2f	multiply(Matrix m);
		void	normalize();
		void	scale(float s0);
		void	set(float x0, float y0);
		void	set(vec2f v0);

		bool operator==(const vec2f &rhs);
		vec2f& operator+=(const vec2f &rhs);
		vec2f& operator-=(const vec2f &rhs);
		vec2f& operator/=(const float &rhs);
		vec2f& operator/=(const vec2f &rhs);
		vec2f& operator*=(const float &rhs);
		vec2f& operator*=(const vec2f &rhs);
		vec2f operator+(const vec2f &rhs);
		vec2f operator-(const vec2f &rhs);		
		vec2f operator*(const float &rhs);
		vec2f operator*(const vec2f &rhs);
		vec2f operator/(const float &rhs);
		vec2f operator/(const vec2f &rhs);

		
		const char* c_str();
	
	private:
		void init(float x0, float y0);
	};

	class vec2i
	{
		friend class boost::serialization::access;
		template<class Archive>
		void serialize(Archive & ar, const unsigned int version)
		{
			ar & x;
			ar & y;
		}
	public:
		int x, y;

		vec2i();
		vec2i(int x0, int y0);
		vec2i(float x0, float y0);

		float	magnitude();
		vec2f	multiply(Matrix m);
		void	scale(int s0);
		void	set(int x0, int y0);
		void	set(vec2i v);

		bool operator==(const vec2i &rhs);
		vec2i& operator+=(const vec2i &rhs);
		vec2i& operator-=(const vec2i &rhs);
		vec2i& operator/=(const float &rhs);
		vec2i& operator*=(const float &rhs);
		vec2i operator+(const vec2i &rhs);
		vec2i operator-(const vec2i &rhs);		
		vec2i operator*(const float &rhs);
		vec2i operator*(const vec2i &rhs);
		
		const char* c_str();

	private:
		void init(int x0, int y0);
	};

	class vec4i
	{
		friend class boost::serialization::access;
		template<class Archive>
		void serialize(Archive & ar, const unsigned int version)
		{
			ar & x;
			ar & y;
			ar & z;
			ar & w;
		}
	public:
		int x, y, z, w;
		
		vec4i();
		vec4i(int x0, int y0, int z0, int w0);
		vec4i(float x0, float y0, float z0, float w0);

		void scale(int s0);
		void set(int x0, int y0, int z0, int w0);
		void set(float x0, float y0, float z0, float w0);
		void set(vec4i v0);

		const char* c_str();
		

	private:
		void init(int x0, int y0, int z0, int w0);
	};

	class Matrix
	{
		friend class boost::serialization::access;
		template<class Archive>
		void serialize(Archive & ar, const unsigned int version)
		{
			ar & m;
		}
	public:

		Matrix();
		float* data();
		void glMultMatrixf();
		void glLoadMatrixf();
		void loadIdentity();
		void multiply(Matrix t);
		void orient(Matrix t);
		void orient(vec3f vx, vec3f vy, vec3f vz);
		void position(vec3f v);
		void position(float x0, float y0, float z0);
		void rotate(float an0);
		void rotate(float anX, float anY, float anZ);
		void rotate(vec3f v);
		void translate(vec3f v);
		void translate(float x0, float y0, float z0);
		void transpose();

		vec3f x();
		vec3f y();
		vec3f z();

		Matrix operator+(const Matrix &rhs);
		Matrix operator-(const Matrix &rhs);		
		
		
	private:
		float m[16];
	};

	class rectf
	{
		friend class boost::serialization::access;
		template<class Archive>
		void serialize(Archive & ar, const unsigned int version)
		{
			ar & top;
			ar & bottom;
			ar & left;
			ar & right;
		}
	public:
		float top, bottom, left, right;

		rectf();
		rectf(int l, int t, int r, int b);
		rectf(float l, float t, float r, float b);

		float area();
		vec2f center();

		void forceInside(int l, int t, int r, int b);
		void forceInside(float l, float t, float r, float b);
		void forceInside(rectf r);

		bool isInside(int l, int t, int r, int b);
		bool isInside(float l, float t, float r, float b);
		bool isInside(rectf r0);
		bool isInside(recti r0);
		bool isInside(vec2i v0);
		bool isInside(vec2f v0);
		bool isInside(float x0, float y0);
		bool isInside(int x0, int y0);

		bool isInside_explicit(int l, int t, int r, int b);
		bool isInside_explicit(float l, float t, float r, float b);
		bool isInside_explicit(rectf r0);
		bool isInside_explicit(recti r0);
		bool isInside_explicit(vec2i v0);
		bool isInside_explicit(vec2f v0);
		bool isInside_explicit(float x0, float y0);
		bool isInside_explicit(int x0, int y0);

		bool overlaps(rectf r0);
		bool overlaps(recti r0);

		void scale(float s0);
		void set(int l, int t, int r, int b);
		void set(float l, float t, float r, float b);
		void set(rectf r0);

		const char* c_str();
		
		rectf& operator*=(const float &rhs);
		rectf& operator/=(const float &rhs);
		

	private:
		void init(float t, float b, float l, float r);
	};


	class recti
	{
		friend class boost::serialization::access;
		template<class Archive>
		void serialize(Archive & ar, const unsigned int version)
		{
			ar & top;
			ar & bottom;
			ar & left;
			ar & right;
		}
	public:
		int top, bottom, left, right;

		recti();
		recti(int l, int t, int r, int b);
		recti(float l, float t, float r, float b);

		int area();
		vec2i center();

		void forceInside(int l, int t, int r, int b);
		void forceInside(float l, float t, float r, float b);
		void forceInside(recti r);

		bool isInside(int l, int t, int r, int b);
		bool isInside(float l, float t, float r, float b);
		bool isInside(rectf r0);
		bool isInside(recti r0);
		bool isInside(vec2i v0);
		bool isInside(vec2f v0);
		bool isInside(float x0, float y0);
		bool isInside(int x0, int y0);

		bool isInside_explicit(int l, int t, int r, int b);
		bool isInside_explicit(float l, float t, float r, float b);
		bool isInside_explicit(rectf r0);
		bool isInside_explicit(recti r0);
		bool isInside_explicit(vec2i v0);
		bool isInside_explicit(vec2f v0);
		bool isInside_explicit(float x0, float y0);
		bool isInside_explicit(int x0, int y0);

		bool overlaps(rectf r0);
		bool overlaps(recti r0);

		void scale(int s0);
		void set(int l, int t, int r, int b);
		void set(float l, float t, float r, float b);
		void set(recti r0);

		const char* c_str();
		

	private:
		void init(int t, int b, int l, int r);
	};

	
	class AxisAngle
	{
		friend class boost::serialization::access;
		template<class Archive>
		void serialize(Archive & ar, const unsigned int version)
		{
			ar & axis;
			ar & angle;
			ar & matrix;
		}
	public:
		vec3f axis;
		float angle;
		Matrix matrix;

		AxisAngle();
		AxisAngle(float angle0, vec3f axis0);
		AxisAngle(float angle0, float x0, float y0, float z0);

		const char* c_str();

		void calcMatrix();
		void normalize();
		void rotate(float angle0);
		//void rotate(vec3f angle0);
		void set(float angle0, vec3f axis0);
		void set(float angle0, float x0, float y0, float z0);
		void set(float angle0);
		void set(vec3f axis0);

	};
	
	class quat4f
	{
		friend class boost::serialization::access;
		template<class Archive>
		void serialize(Archive & ar, const unsigned int version)
		{
			ar & w;
			ar & x;
			ar & y;
			ar & z;
			ar & matrix;
		}
	public:
		float	w;
		float	x;
		float	y;
		float	z;
		Matrix	matrix;

		quat4f();
		quat4f(float w0, float x0, float y0, float z0);
		quat4f(float w0, vec3f v0);
		quat4f(AxisAngle a0);

		const char* c_str();

		vec3f axis();
		void calcMatrix();
		vec3f euler();
		float magnitude();
		quat4f multiply(float rhs);
		quat4f multiply(quat4f rhs);
		void normalize();
		void set(float w0, vec3f v0);
		void set(float w0, float x0, float y0, float z0);
		void set(AxisAngle a0);
		void set(quat4f q0);

		quat4f& operator+(const quat4f &rhs);
		quat4f& operator+=(const quat4f &rhs);
		quat4f& operator*=(const quat4f &rhs);
		quat4f operator*(const float &rhs);
		quat4f operator*(const quat4f &rhs);
	};

	class Diffeq3f
	{
	public:
		vec3f g;
		vec3f mass;
		vec3f position;
		vec3f velocity;
		vec3f forces;
		std::vector<float> k_values;
		std::vector<vec3f> k_targets;

		Diffeq3f();
		void init();

		void addForce(vec3f f);
		void addSpring(float ck, vec3f target);
		
		void clearForces();
		void clearSprings();
		
		void setDampening(float d);
		void setDampening(vec3f d);
		void setForces(vec3f f);
		void setMass(float m);
		void setMass(vec3f m);
		void setPosition(vec3f s);
		void setVelocity(vec3f v);

		void solve(float dt);
	};

	void DESolve(std::vector<Diffeq3f> &stack, float dt);

	float rnd();
	float nrnd();
	float sgn(float f0);

}}


#endif __BASECODE_MATH_H__
