// Program:  Model.h - 3D Model Class
// Language: C++
// Author:   Daniel Kennedy (Syn9)
//
// Copyright (c) 2009-2010
// ____________________________________________________________________________
#ifndef INC_MODEL_H
#define INC_MODEL_H

#include <GL/glew.h>
#define GLFW_DLL
#include <GL/glfw.h>
#include <list>
#include <vector>
#include <fstream>
#include <string>
#include <algorithm>
#include "Math.h"
#include "Color.h"

namespace basecode
{
	using namespace Math;
		
	struct Texcoord
	{
		vec2f	uv[3];
	};

	struct Face
	{
		unsigned int	point_id[3];
		unsigned int	uv_id[3];
		unsigned int	npoints;
		Texcoord		texcoord[4];
		vec2f			nmapcoord;
		vec3f			vnormals[3];
		Color			vcolors[3];
		vec2f			uvs[3];
		vec3f			normal;
		std::string		materialName;
		unsigned int	materialID;
		float			d;
	};

	struct Material
	{
		std::string		name;
		Color			color;
		Color			emitting;
		std::string		textureStr;
		unsigned int	textureID;
	};

	class Model
	{
	private:
		float			vertexSnapDistance;

	public:
		std::string		filename;
		
		std::vector<vec3f>		vertices;
		std::vector<Face>		faces;
		std::vector<vec3f>		normals;
		std::vector<Material>	materials;
		std::vector<vec2f>		texcoords;

		vec3f			meanVertex;

		unsigned int vertexBuffer;
		unsigned int colorBuffer;

		
		Model();
		~Model();

		float			area2(Face t);

		void			calc_normal(unsigned int face);
		void			calc_normals();
		void			center();
		vec3f			centroid();
		vec3f			fcentroid();
		void			clear ();
		void			convex_hull(unsigned int max_faces);
		unsigned int	containment_count();
		float			surface_area();
		float			surface_area2();
		void			prune_vertices();
		std::vector<vec2i> edge_list();

		bool			load (std::string file);
		bool			load_obj (std::string file);
		void			load_mtl (std::string file);

		unsigned int	newNormal (float x, float y, float z);
		unsigned int	newNormal (vec3f n);
		
		unsigned int	newFace ();
		unsigned int	newFace (vec3f p1, vec3f p2, vec3f p3);
		unsigned int	newFace (unsigned int p1, unsigned int p2, unsigned int p3);
		unsigned int	newFace (vec3f p1, vec3f p2, vec3f p3, Color c);
		unsigned int	newFace (Face t);
		unsigned int	newFace_explicit (vec3f p1, vec3f p2, vec3f p3);
		unsigned int	newFace_explicit (vec3f p1, vec3f p2, vec3f p3, Color c);

		unsigned int	newUV (float u, float v);
		unsigned int	newUV (vec2f uv);

		unsigned int	newVertex (float x, float y, float z);
		unsigned int	newVertex (vec3f v);
		unsigned int	newVertex_explicit (float x, float y, float z);
		unsigned int	newVertex_explicit (vec3f v);

		void			normalize();

		void			orient_normals(float direction);
		float			radius();
		float			minRadius();
		void			removeCreases();

		float			getVertexSnapDistance();
		void			setVertexSnapDistance(float d);

		void			scale(float x0, float y0, float z0);
		void			scale(float s0);

		void genBuffers();
		void buildBuffers();
		void color(float r, float g, float b);
		

		void str_explode(std::string haystack, std::string needle, std::vector<std::string> *result);
		
	};
	

	class ModelHandler
	{
	private:
		std::list<Model>	modelList;

	public:
		Model*	newModel();
		void	deleteModel(Model* model);

	};

}


#endif
